/*---------------------------------------------------------------------------*\

    FILE....: VPBAPI.CPP
    TYPE....: C++ Module
    AUTHOR..: John Kostogiannis and David Rowe
    DATE....: 4/3/98

    This file contains the implementation of several API functions, and
    also quite a few "helper" functions that are used by API functions
    in other files.  

    Can be considered the "main" of the VPBAPI library.
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

				 INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>

#include "adpcm.h"
#include "contypes.h"
#include "config.h"
#include "comm.h"
#include "fifo.h"
#include "mess.h"
#include "vpbreg.h"
#include "wobbly.h"
#include "vpbhandle.h"
#include "mapdev.h"
#include "vpbapi.h"
#include "apifunc.h"
#include "playrec.h"
#include "pip.h"
#include "digits.h"
#include "call.h"
#include "vpbdial.h"
#include "vpbtoned.h"
#include "vpbtimer.h"
#include "vpbagc.h"
#include "vpbvox.h"
#include "comp.h"
#include "objtrack.h"
#include "generic.h"
#include "coff.h"
#include "envtone.h"
#include "verbose.h"
extern "C" {
#include "toned.h"
}
 
/*---------------------------------------------------------------------------*\

				 DEFINES
							
\*---------------------------------------------------------------------------*/

#define MESSEVENTDELAY	20		// how often VPBs are polled by driver
#define APIQUEUESIZE	128		// max num events in API Q
#define MASK_EVENTS	9		// number of maskable API events

#define	SIZE_OF_VPB_EVENT_WORDS	(sizeof(VPB_EVENT)/sizeof(word))

// Call progress state machine event codes.
// These codes are passed up in the message from the DSP when a CP tone
// is detected.

#define	toned_DIAL	0		// dial tone
#define	toned_RING	1		// ring
#define	toned_BUSY	2		// busy

/*---------------------------------------------------------------------------*\

			               GLOBALS
							
\*---------------------------------------------------------------------------*/

int  model;			// which platform we are using
Comm *vpb_c;			// ptr to comm object
static Fifo *APIQ;	        // ptr to API event Q object
static int Init = 0;		// indicates API initialised
static USHORT numboards;	// number of boards in system (from reg)
static USHORT Totalchns;	// total channels in system
static USHORT sleepms;		// sleep period for threads

static int MMQ_flag;		// Signals MMQ to finish

VPB_DEV	*vpb_dev;		// array of device information

static int bayonne_mode;        // non-zero if driver in "bayonne" mode

// Critical section for put_evt, necessary because the driver via the
// polling timer may call putevt at the same time as the application.

GENERIC_CRITICAL_SECTION	PutEvtSect;
GENERIC_CRITICAL_SECTION	VpbOpenSect;

// Off and On hook messages

static word offhook[] = {
	PC_LCODEC_OFFHK,
	PC_CODEC_OFFHK,
	0					// channel
};

static word onhook[] = {
	PC_LCODEC_ONHK,
	PC_CODEC_ONHK,
	0					// channel
};

// Tone detector params describing Australian/Studio 308/US dial tone 

static VPB_DETECT toned_dial = {
	2,			// number of cadence states
	VPB_DIAL,		// tone id
	1,			// number of tones
	400,			// freq1
	100,			// bandwidth1
	0,			// freq2, N/A
	0,			// bandwidth2, N/A
	-40,			// level1
	0,			// level2, N/A
	0,			// twist, N/A
	10,			// SNR
	40,			// glitchs of 40ms ignored

	
	{
		{
			RISING,
			0,
			0,
			0
		},
	
		{
			TIMER,			// state 1
			3000,                   // 2s to avoid mix-up with US
			0,                      // US ringback
			0
		}
	}
};

// Studio 308 ringback

static VPB_DETECT toned_ringback_308 = {
	2,		  // number of cadence states
	VPB_RINGBACK_308, // tone id
	1,		// number of tones
	425,		// freq1
	100,		// bandwidth1
	0,		// freq2, N/A
	0,	        // bandwidth2, N/A
	-20,		// level1
	0,		// level2, N/A
	0,		// twist, N/A
	10,		// SNR
	40,		// glitchs of 40ms ignored
	
	{
		{
			VPB_RISING,	// state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	// state 1			
			0,
			800,		// ton min		
			1200		// ton max		
		}
	}
};

// message describing state transitions for Studio 308 busy 

static VPB_DETECT toned_busy_308 = {
	3,		// number of states		
	VPB_BUSY_308,	// tone id
	1,		// number of tones		
	425,		// f1:  centre frequency	
	100,		// bw1: bandwidth		
	0,		// f2: N/A			
	0,		// bw2: N/A			
	-20,		// A1: -10 dbm0			
	0,		// A2: N/A 			
	0,		// twist: N/A			
	10,		// SNR: 10dB			
	40,		// glitchs of 40ms ignored

	{
		{
			VPB_RISING,	// state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	// state 1			
			0,
			450,		// 450ms ton min		
			550		// 550ms ton max		
		},

		{
			VPB_RISING,	// state 2			
			0,
			450,		// 450ms toff min		
			550		// 550ms toff max		
		}
	}
};

// message describing state transitions for Australian Busy
#ifdef TMP

static VPB_DETECT toned_busy_aust = {
	3,		 // number of states		
	VPB_BUSY_AUST,   // tone id
	1,		 // number of tones		
	425,		 // f1:  centre frequency	
	100,		 // bw1: bandwidth		
	0,		 // f2: N/A			
	0,		 // bw2: N/A			
	-20,		 // A1: -10 dbm0			
	0,		 // A2: N/A 			
	0,		 // twist: N/A			
	10,		 // SNR: 10dB			
	40,		 // glitchs of 40ms ignored

	{
		{
			VPB_RISING,	 // state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	 // state 1			
			0,
			300,					
			450
		},					

		{
			VPB_RISING,	 // state 2			
			0,
			300,					
			450
		},					
	}
};
#endif

// grunt detector, looks for wide band energy between 600 and 1400 Hz

static VPB_DETECT toned_grunt = {
	3,			// number of states		
	VPB_GRUNT,		// tone id
	1,			// number of tones		
	2000,			// f1:  centre frequency	
	3000,			// bw1: bandwidth		
	0,			// f2: N/A			
	0,			// bw2: N/A			
	-40,			// A1: dbM0		
	0,			// A2: N/A 			
	0,			// twist: N/A			
	0,			// SNR: 10dB			
	40,			// glitchs of 40ms ignored

	{
		{
			VPB_DELAY,		// state 0		
			1000,
			0,
			0
		},
		
		{
			VPB_RISING,		// state 1
			0,
			40,
			0
		},

		{
			TIMER,			// state 2
			100,
			0,
			0
		}
	}

};

// US PSTN ringback

static VPB_DETECT toned_ringback_us = {
	2,		// number of cadence states
	VPB_RINGBACK,	// tone id
	1,		// number of tones
	425,		// freq1
	200,		// bandwidth1
	0,		// freq2, N/A
	0,	        // bandwidth2, N/A
	-20,		// level1
	0,		// level2, N/A
	0,		// twist, N/A
	10,		// SNR
	40,		// glitchs of 40ms ignored
	
	{
		{
			VPB_RISING,	// state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	// state 1			
			0,
			1600,		// ton min		
			2400		// ton max		
		}
	}
};

// US busy tone

static VPB_DETECT toned_busy_us = {
	3,		 // number of states		
	VPB_BUSY,        // tone id
	2,		 // number of tones		
	480,		 // f1:  centre frequency	
	100,		 // bw1: bandwidth		
	620,		 // f2: N/A			
	100,		 // bw2: N/A			
	-20,		 // A1: -10 dbm0			
	-20,		 // A2: N/A 			
	10,		 // twist: N/A			
	10,		 // SNR: 10dB			
	40,		 // glitchs of 40ms ignored

	{
		{
			VPB_RISING,	 // state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	 // state 1			
			0,
			450,					
			550
		},					

		{
			VPB_RISING,	 // state 2			
			0,
			450,					
			550
		},					
	}
};

// Tone detector params describing Fax CNG/preamble

static VPB_DETECT toned_fax = {
	6,			// number of cadence states
	VPB_FAX,		// tone id
	1,			// number of tones
	1800,			// freq1
	400,			// bandwidth1
	0,			// freq2, N/A
	0,			// bandwidth2, N/A
	-40,			// level1
	0,			// level2, N/A
	0,			// twist, N/A
	10,			// SNR
	40,			// glitchs of 40ms ignored

	
	{
		{
			RISING,
			0,
			0,
			0
		},
	
		{
			TIMER,			// state 1
			1000,                   
			0,                     
			0
		}
	}
};

/*-------------------------------------------------------------------------*\

			     LOCAL FUNCTION HEADERS

\*-------------------------------------------------------------------------*/

void ConfigureVPB(Comm *c, USHORT numboards);
void CloseVPB(Comm *c, USHORT numboards);
void MonitorMessageQueue(void *data);
void start_event_callback_thread(VPB_EVENT *e);
void event_callback_thread(void *data);

void ProcessVPBMessage(word mess[], USHORT board);
void GenerateTONEDEvent(word mess[]);
void ProcessPlayandRec();

void StartMonitorMessageQueue();
void KillMonitorMessageQueue();
void ConfigVPB4(Comm *c,	int	b);
void ConfigVPB8L(Comm *c, int b);
void set_codec_reg(int    chdev, // channel handle
		   USHORT addr,  // 8-bit address of codec register
		   USHORT data,  // 8-bit data to write to register
		   Comm   *c);

/*-------------------------------------------------------------------------*\

			       FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_start()
	AUTHOR...: David Rowe
	DATE.....: 24/8/00

	Initialises the API software (boots DSPs etc) when operating
	in "bayonne" mode.  Calling this function before vpb_open() starts
	the driver in "bayonne" mode.  If vpb_open() is called without 
	calling this function first, the driver is started in "regular" 
	mode.

	The major feature of bayonne mode is that the user must set up
	a thread to periodically call (poll) the vpb_get_event_async_bayonne()
	function.  Without this thread the driver will not function.  This
	has certain advantages where the user whishes to control the number
	of threads in the system, and was specifically written for the
	bayonne package (www.bayonne.cx).  See the unittest tvpb.cpp for an
	example of a program implemented using bayonne mode.

	In regular mode, the driver sets up all sorts of polling threads
	which leads to generally simpler application code.  In other words
	the choice of bayonne vs regular mode is the usual choice of 
	flexibility vs complexity :-).  Most of the programs in the unittest
	directory use regular mode.

	Note: bayonne mode was originally supported by the 2.0.x series
	vpb drivers, and regular mode by the 1.x.y series of drivers.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_start(int num_cards, int *bases, char *firmware_file, int model)
{
	VPBREG	*pvpbreg;
	int	i;

	try {
	
	// Initialise if first time into this function

	if (!Init) {
	        bayonne_mode = 1;
		vpb_config(num_cards, bases, firmware_file, model);
		Init = 1;
		sleepms = 20;
			
		mprintf("Initialising Driver\n...");

		// Open comms, this:
		// 1. boots DSP on each VPB card
		// 2. reads parameters from the windows registry into the
		//    VPB registry structure.
		// 3. downloads vpb reg parameters to each DSP.

		// Determine total number of channels
	
		CheckNew(vpb_c = new Comm());
		Totalchns = 0;
		numboards = vpb_c->numboards();
		for(i=0;i<numboards;i++){
			pvpbreg = vpb_c->vpbreg(i);   // reg for each card
			Totalchns += pvpbreg->numch;
			pvpbreg++;
		}
		assert(Totalchns);
		model = vpb_c->vpbreg(0)->model;
                printf("Bds:%d   Chans:%d\n",numboards,Totalchns);

		CheckNew(vpb_dev = new VPB_DEV[Totalchns]);

		// Initialise device handle flags 
		// Initialise ch. event masks to enable all maskable events
		
		for(i=0;i<Totalchns;i++) {
			vpb_dev[i].DevHndles = VPB_OFF;
			vpb_dev[i].Chnevtmsk = 0xffff;	// all enabled
		}

		// Now configure each channel of each VPB.
		// This sets up the signal processing modules such as
		// FIFOs, codec I/O, etc

		ConfigureVPB(vpb_c, numboards);  // DSP signal processing
      		playrec_open(Totalchns);         // play and record
		vpbdial_open(Totalchns);         // tone generation

		if (vpb_c->vpbreg(0)->model == VPB8L)
			pip_open(Totalchns);
		digits_open(Totalchns);
		call_open(Totalchns);
			
		// Initialse critical sections

		GenericInitializeCriticalSection(&PutEvtSect);

		// Ensure all ports stay onhook until requested
			
		if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
		    for(i=0; i<Totalchns; i++)
			vpb_set_codec_reg(i, 0x12, 0x1c);

		mprintf("Driver initialised OK!\n");
	}

	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_open"));
	}
	
	return VPB_OK;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_open()
	AUTHOR...: John Kostogiannis
	DATE.....: 26/11/97

	Opens a VPB device (channel), and returns a handle to that device.
	
	See also vpb_start().

\*--------------------------------------------------------------------------*/

int WINAPI vpb_open(unsigned int board, unsigned int channel)
{
	VPBREG	*pvpbreg;
	int	devhandle,i;

	try {
	
		// Initialise if first time into this function

		if (!Init) {
			mess_init();
		        bayonne_mode = 0;
			Init = 1;
			
			mprintf("Initialising Driver\n...");

			// Open comms, this:
			// 1. boots DSP on each VPB card
			// 2. reads parameters from the windows registry into the
			//    VPB registry structure.
			// 3. downloads vpb reg parameters to each DSP.

			// Determine total number of channels
	
			CheckNew(vpb_c = new Comm());
			Totalchns = 0;
			numboards = vpb_c->numboards();
			for(i=0;i<numboards;i++){
				pvpbreg = vpb_c->vpbreg(i);		// reg for each card
				Totalchns += pvpbreg->numch;
				pvpbreg++;
			}
			assert(Totalchns);
			model = vpb_c->vpbreg(0)->model;

			CheckNew(vpb_dev = new VPB_DEV[Totalchns]);

			// Now configure each channel of each VPB.
			// This sets up the signal processing modules such as
			// FIFOs, codec I/O, dtmf and call progress detection etc

			ConfigureVPB(vpb_c, numboards);
			
			// Initialise device handle flags  
		
			for(i=0;i<Totalchns;i++)
				vpb_dev[i].DevHndles = VPB_OFF;
						
			// Initialise ch. event masks to enable all maskable events
			
			for(i=0;i<Totalchns;i++)
				vpb_dev[i].Chnevtmsk = 0xffff;	// all enabled

			if (vpb_c->vpbreg(0)->model == VPB8L)
				pip_open(Totalchns);
			playrec_open(Totalchns);

			// Init tone generator

			vpbdial_open(Totalchns);

			// Init timer module

			vpbtimer_open();

			if (vpb_c->vpbreg(0)->model == VPB8L) {
				vpbvox_open(Totalchns);
				vpbagc_open(Totalchns);
			}

			// DR 24/2/03 - enable VOX for OpenLine4
			//if (vpbreg_get_model() == VPB_V4LOG) {
			vpbvox_open(Totalchns);
			//}

			sleepms = 20;

			// init event callback function ptrs to no callbacks

			for(i=0; i<Totalchns; i++)
				vpb_dev[i].event_callback = NULL;

			// init ring module

			digits_open(Totalchns);
			call_open(Totalchns);
			
			// Initialise API event Q

			CheckNew(APIQ = new Fifo(APIQUEUESIZE*SIZE_OF_VPB_EVENT_WORDS));
			
			// Init API Q for each channel

			for(i=0;i<Totalchns;i++)
				CheckNew(vpb_dev[i].APIQ = new Fifo(APIQUEUESIZE*SIZE_OF_VPB_EVENT_WORDS));

			// Initialse critical sections

			GenericInitializeCriticalSection(&PutEvtSect);

			// activate the Monitoring of the Message Queue 

			StartMonitorMessageQueue();			
			
			// Ensure all ports stay onhook until requested
			
			if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
			    for(i=0; i<Totalchns; i++)
				vpb_set_codec_reg(i, 0x12, 0x1c);
			
			mprintf("Driver initialised OK!\n");
		}

		// no way code should reach this point without Initialisation

		assert(Init);

		// OK, regular device handle allocation starts here

		// check board and channel for validity

		if (board > numboards)
			throw Wobbly(VPBAPI_INVALID_BOARD_NUMBER);
		if (model == VPB4) {
			if (channel > 4)
				throw Wobbly(VPBAPI_INVALID_CHANNEL_NUMBER);
		}
		if (model == VPB8L) {
			if (channel > 8)
				throw Wobbly(VPBAPI_INVALID_CHANNEL_NUMBER);
		}
		
		// check to see if handle already open

		VPBHandle(board,channel,&devhandle,vpb_c->vpbreg(0),numboards);
		if (vpb_dev[devhandle].DevHndles == VPB_ON)
			throw Wobbly(VPBAPI_CHANNEL_ALREADY_OPEN);

		vpb_dev[devhandle].DevHndles = VPB_ON;

		mprintf("Channel device [%d] initialised OK!\n", devhandle);

	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_open"));
	}
	
	return(devhandle);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_close()
	AUTHOR...: John Kostogiannis
	DATE.....: 27/11/97

	Closes a device.  If this is the last device remaining, then the
	API is closed down (memory free-ed, DSP stopped etc).

\*--------------------------------------------------------------------------*/

int WINAPI vpb_close(int devhandle)
{

	int	NoDevHndles;
	int i;
	
	try {
		// validate argument

		ValidHandleCheck(devhandle);

		// close device
		
		vpb_dev[devhandle].DevHndles = VPB_OFF;

		// Determine if any devices are still open
		
		NoDevHndles = 1;

		for(i=0;i<Totalchns;i++){
			if(vpb_dev[i].DevHndles == VPB_ON)
				NoDevHndles = 0;
		}

		// If no devices are open, close down

		if (NoDevHndles && !bayonne_mode) {

			mprintf("Closing down driver\n");
			
			KillMonitorMessageQueue();
			CloseVPB(vpb_c, numboards);
			vpbtoned_close();
			objtrack_close();
			if (vpb_c->vpbreg(0)->model == VPB8L) {
				vpbagc_close();
			}
			if (vpb_c->vpbreg(0)->model == VPB8L)
				pip_close();
			vpbvox_close();
			vpbdial_close();
			vpbtimer_close();
			playrec_close();
			digits_close();
			call_close();
			delete APIQ;
			for(i=0;i<Totalchns;i++)
				delete vpb_dev[i].APIQ;
			delete [] vpb_dev;
			delete vpb_c;
			GenericDeleteCriticalSection(&PutEvtSect);
			Init = 0;

			mprintf("Driver closed down OK!\n");
			mess_mprintf_off();
		}
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_close"));
	}

	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_stop()
	AUTHOR...: David Rowe
	DATE.....: 24/8/00

	API is closed down (memory free-ed, DSP stopped etc).
	Only used for "bayonne" mode.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_stop()
{
	assert(bayonne_mode);
	mprintf("Closing down driver\n");
			
	CloseVPB(vpb_c, numboards);
	vpbtoned_close();
	objtrack_close();
	vpbdial_close();
	playrec_close();
	delete [] vpb_dev;
	delete vpb_c;
	Init = 0;

	mprintf("Driver closed down OK!\n");
	mess_mprintf_off();
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: ValidHandleCheck()
	AUTHOR...: John Kostogiannis
	DATE.....: 23/12/97

	Check for a valid device handle.  Handle is valid if device has been
	opened.  If handle is invalid, exception is thrown.

\*--------------------------------------------------------------------------*/

void ValidHandleCheck(int devhandle)
// int	devhandle	Device handle to validate
{	
	// range check handle

	if ((devhandle < 0) || (devhandle >= Totalchns))
		throw Wobbly(VPBAPI_INVALID_HANDLE);

	// check if handle referes to open device

	if (vpb_dev[devhandle].DevHndles == VPB_OFF)
		throw Wobbly(VPBAPI_INVALID_HANDLE);
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	ConfigureVPB 
	AUTHOR..:	David Rowe
	DATE....:	6/2/98
	
	This function is called during system initialisation to create and
	connect the configuration objects that perform the signal processing
	operations for each channel.

	This function should be called before the timer VPB polling function
	MonitorMessageQueue() is started, as this function directly reads the
	VPB message queue.

\*-------------------------------------------------------------------------*/

void ConfigureVPB(Comm *c, USHORT numboards)
//      Comm    *c;		ptr to comm object for VPB(s)
//	USHORT  numboards;	number of boards in system
{
	VPBREG	*v;		// ptr to reg info for current VPB
	int	b,ch;		// board
	int     handle;

	mprintf("Configuring VPBs...\n");

	objtrack_open();

	// Init tone detector
			
	vpbtoned_open(Totalchns);

	// now set up config on each channel device

	for(b=0; b<numboards; b++) {
		v = c->vpbreg(b);

		switch(v->model) {
		case VPB4:
			ConfigVPB4(c, b);
			break;
		case VPB8L:
			ConfigVPB8L(c, b);
			break;
		case VPB_V12PCI:
			for(ch=0; ch<v->numch; ch++) {
				handle = mapdevtohndle(b,ch);
				// set default prog tone dets,

				settonedet(handle, &toned_dial);
				settonedet(handle, &toned_ringback_us);
				settonedet(handle, &toned_busy_us);
				settonedet(handle, &toned_grunt);
				settonedet(handle, &toned_ringback_308);
				settonedet(handle, &toned_busy_308);
				settonedet(handle, &toned_fax);

				// look for tones specified by env variables

				envtone_read_tones_from_env(handle);
			}
			break;
		default:
			assert(0);
		}

		// OK, start config manager for this board...
		
		config_run(c, b);

		// now start host echo canceller threads (if enabled)
		
		#ifdef LINUX
		if (v->echo_mode == DSPFIFO_ECHO) {
			for(ch=0; ch<v->numch; ch++) {
				v->ec[ch] = dspfifo_echo_start_thread(b, ch);
			}
		}
		#endif
	}

	// override default prog tones from reg

	if (c->vpbreg(0)->model == VPB4)
		vpbreg_load_default_tones(Totalchns);

	mprintf("VPBs configured OK!\n");
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	ConfigVPB4
	AUTHOR..:	David Rowe
	DATE....:	17/6/98
	
	Configures all channels of a 4 channel VPB card.

\*-------------------------------------------------------------------------*/

void ConfigVPB4(
	Comm	 *c,		// comm ptr
	int	 b		// board 		
)
{
 	int	 ch;		// channel
	int	 obj;		// object counter
	int	 handle;	// handle of this channel
	VPBREG	 *v;		// reg info for this card
	int	 stobj;		// start object for this channel

	obj = 0;
	v = c->vpbreg(b);
	
	for(ch=0; ch<v->numch; ch++) {
		handle = mapdevtohndle(b,ch);
	
		mprintf("Configuring dev: %d  brd: %d  ch: %d\n",handle,b,ch);
	
		// Up (channel to PC) signal processing objects

		mprintf(" dev: %d :Up channel objects, obj:%d\n",handle, obj);

		objtrack_add_object(UPOBJ, obj, handle, b);
		config_create_object(c, b, CODEC_AD , obj++, ch, 0);	// 0
		config_create_object(c, b, ALAW2LIN , obj++, ch, 0);	// 1
		objtrack_add_object(ECHOOBJ, obj, handle, b);
		config_create_object(c, b, ECHO     , obj++, ch, 0);	// 2
		objtrack_add_object(DTMFOBJ, obj, handle, b);
		config_create_object(c, b, DTMF     , obj++, ch, 0);	// 3
		objtrack_add_object(CPOBJ, obj, handle, b);
		config_create_object(c, b, TONED    , obj++, ch, 0);	// 4
		config_create_object(c, b, LIN2ALAW , obj++, ch, 0);	// 5
		config_create_object(c, b, LIN2MULAW, obj++, ch, 0);	// 6
		config_create_object(c, b, LIN2ADPCM, obj++, ch, 0);	// 7
		objtrack_add_object(VOXOBJ, obj, handle, b);
		config_create_object(c, b, VOX	    , obj++, ch, 0);	// 8
		config_create_object(c, b, PACK	    , obj++, ch, 0);	// 9
		objtrack_add_object(FIFOUPOBJ, obj, handle, b);
		config_create_object(c, b, PACKED_FIFO_UP, obj++, ch, 
				     v->szrxdf[ch]);	               // 10
		
		// Down (PC to channel) signal processing objects

		mprintf(" dev: %d :Down channel objects, obj:%d\n",handle,obj);

		objtrack_add_object(FIFODNOBJ, obj, handle, b);
		objtrack_add_object(DNOBJ, obj, handle, b);
		config_create_object(c, b, PACKED_FIFO_DOWN, obj++, ch, 
				     v->sztxdf[ch]);	                // 0
		config_create_object(c, b, UNPACK, obj++, ch, 0);	// 1
		objtrack_add_object(TONEOBJ, obj, handle, b);
		config_create_object(c, b, TONEG    , obj++, ch, 0);	// 2
		config_create_object(c, b, MULAW2LIN , obj++, ch, 0);	// 3

		/* DR 8/3/02 - loose this, replace with adder to keep obj #
		   the same
		config_create_object(c, b, ADPCM2LIN , obj++, ch, 0);	// 5
		*/
		config_create_object(c, b, ADDER , obj++, ch, 0);	// 4

		config_create_object(c, b, LIN2ALAW , obj++, ch, 0);	// 5
		config_create_object(c, b, CODEC_DA , obj++, ch, 0);	// 6
		config_create_object(c, b, ALAW2LIN , obj++, ch, 0);	// 7
		objtrack_add_object(DELAYOBJ, obj, handle, b);
		config_create_object(c, b, DELAY    , obj++, ch, 0);	// 8

		// Up side wiring 

		stobj = objtrack_handle_to_id(UPOBJ, handle);
		config_create_wire(c, b, stobj+0, stobj+1);
		config_create_wire(c, b, stobj+1, stobj+2);
		if (v->echo_mode == DSPFIFO_ECHO) {
			// Host based echo canceller
			mprintf("Host based echo canceller\n");
			config_create_wire(c, b, stobj+1, stobj+3);
			config_create_wire(c, b, stobj+1, stobj+4);
			config_create_wire(c, b, stobj+1, stobj+5);
			config_create_wire(c, b, stobj+1, stobj+6);
			config_create_wire(c, b, stobj+1, stobj+7);
			config_create_wire(c, b, stobj+1, stobj+8);
			config_create_wire(vpb_c, b, stobj+1, stobj+10);
		}
		else {
			// DSP based echo canceller
			config_create_wire(c, b, stobj+2, stobj+3);
			config_create_wire(c, b, stobj+2, stobj+4);
			config_create_wire(c, b, stobj+2, stobj+5);
			config_create_wire(c, b, stobj+2, stobj+6);
			config_create_wire(c, b, stobj+2, stobj+7);
			config_create_wire(c, b, stobj+2, stobj+8);
			config_create_wire(vpb_c, b, stobj+2, stobj+10);
		}
		config_packrate(vpb_c, b, stobj+10, 1);

		// Down side wiring

		stobj = objtrack_handle_to_id(DNOBJ, handle);
		config_create_wire(c, b, stobj+0, stobj+1);    // FIFO->UNPACK
		config_create_wire(c, b, stobj+1, stobj+3);
		config_create_wire(c, b, stobj+5, stobj+7);
		// DR 8/3/02 config_create_wire(c, b, stobj+1, stobj+5);
		config_create_wire(c, b, stobj+4, stobj+8);
		//config_create_wire(c, b, stobj+5, stobj+8);
		/* DR 8/3/02
		config_connect_to_zerobuf(c, b, stobj+6);
		config_connect_to_zerobuf(c, b, stobj+8);
		*/
		// DR 8/3/02 - set up linear
		#ifdef LIN1
		config_create_wire(vpb_c, b, stobj+0, stobj+6);
		config_create_wire(vpb_c, b, stobj+0, stobj+8);
		config_packrate(vpb_c, b, stobj, 1);
		#else
		config_create_wire(vpb_c, b, stobj+0, stobj+4);
		config_create_wire(vpb_c, b, stobj+2, stobj+4);
		config_create_wire(vpb_c, b, stobj+4, stobj+5);
		config_create_wire(vpb_c, b, stobj+5, stobj+6);
		config_packrate(vpb_c, b, stobj, 1);
		#endif

		// connect echo canceller

		int dest = objtrack_handle_to_id(ECHOOBJ, handle);
		stobj = objtrack_handle_to_id(DELAYOBJ, handle);
		config_create_wire(c, b, stobj, dest);

		// Configure call progress detector for this channel

		mprintf(" dev: %d :call Progress state machines\n",handle);

		// set default prog tone dets,

		settonedet(handle, &toned_dial);
		settonedet(handle, &toned_ringback_us);
		settonedet(handle, &toned_busy_us);
		settonedet(handle, &toned_grunt);
		settonedet(handle, &toned_ringback_308);
		settonedet(handle, &toned_busy_308);
		settonedet(handle, &toned_fax);

		// look for tones specified by environment variables

		envtone_read_tones_from_env(handle);

		// Disable UP objects

		stobj = objtrack_handle_to_id(UPOBJ, handle);
		#define OLD1
		#ifdef OLD1
		if (v->echo_mode == DSPFIFO_ECHO) {
			// disable DSP echo canceller if host ec enabled
			config_disable_object(c, b, stobj+2);	// ECHO
		}
		//config_disable_object(c, b, stobj+4);	// TONED
		config_disable_object(c, b, stobj+5);	// ALAW
		config_disable_object(c, b, stobj+6);	// MULAW
		config_disable_object(c, b, stobj+7);	// ADPCM
		if (vpbreg_get_model() != VPB_V4LOG) {
		  config_disable_object(c, b, stobj+8);	// VOX
		}
		config_disable_object(c, b, stobj+9);	// PACK
		//config_disable_object(c, b, stobj+10);	// FIFO
		#endif
	
		// Disable down objects except D/A, LIN2ALAW, and DELAY
		#define OLD2
		#ifdef OLD2
		stobj = objtrack_handle_to_id(DNOBJ, handle);
		//config_disable_object(c, b, stobj);	// FIFO
		config_disable_object(c, b, stobj+1);	// UNPACK
		//config_disable_object(c, b, stobj+2);	// TONEG
		config_disable_object(c, b, stobj+3);	// ALAW
		//config_disable_object(c, b, stobj+4);	// ADDER
		config_disable_object(c, b, stobj+7);	// MULAW
		//config_disable_object(c, b, stobj+5);	// ADPCM
		#endif
		
	}
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	ConfigVPB8L
	AUTHOR..:	David Rowe
	DATE....:	17/6/98
	
	Configures all channels of a 8 channel logging VPB card.

\*-------------------------------------------------------------------------*/

void ConfigVPB8L(
	Comm	*c,			// comm ptr
	int		b			// board 										
	)
{
 	int		ch;			// channel
	int		obj;		// object counter
	int		handle;		// handle of this channel
	VPBREG	*v;			// reg info for this card
	int		stobj;		// start object for this channel
	int		adobj;		// adc object for this channel

	obj = 0;
	v = c->vpbreg(b);
	
	// configure ADs first, as this makes them get processed first
	// which allows us to have occasional (not average) computational
	// loads greater than the frame length

	for(ch=0; ch<v->numch; ch++) {
		handle = mapdevtohndle(b,ch);
		mprintf("Configuring dev: %d  brd: %d  ch: %d\n",handle,b,ch);
		objtrack_add_object(ADCOBJ, obj, handle, b);
		config_create_object(c, b, ADC_24K, obj++, ch, 0);
	}
	
	// now configure rest of objects

	for(ch=0; ch<v->numch; ch++) {
		handle = mapdevtohndle(b,ch);
		mprintf("Configuring dev: %d  brd: %d  ch: %d\n",handle,b,ch);
	
		objtrack_add_object(UPOBJ, obj, handle, b);
//		config_create_object(c, b, DEC24K_TO_8K			 , obj++, ch, 0);	// 0
		config_create_object(c, b, FIRD			 , obj++, ch, 0);	// 0
		objtrack_add_object(AGCOBJ, obj, handle, b);
		config_create_object(c, b, AGC		  	 , obj++, ch, 0);	// 1
		objtrack_add_object(VOXOBJ, obj, handle, b);
		config_create_object(c, b, VOX			 , obj++, ch, 0);	// 2
		objtrack_add_object(DTMFOBJ, obj, handle, b);
		config_create_object(c, b, DTMF			 , obj++, ch, 0);	// 3
		config_create_object(c, b, LIN2ADPCM	 , obj++, ch, 0);	// 4
		config_create_object(c, b, PACK			 , obj++, ch, 0);	// 5
		objtrack_add_object(FIFOUPOBJ, obj, handle, b);
		config_create_object(c, b, PACKED_FIFO_UP, obj++, ch, v->szrxdf[ch]);	// 6
		config_create_object(c, b, LIN2MULAW	 , obj++, ch, 0);	// 7
		config_create_object(c, b, LIN2ALAW		 , obj++, ch, 0);	// 8
		config_create_object(c, b, DEC24K_TO_8K	 , obj++, ch, 0);	// 9
		
		adobj = objtrack_handle_to_id(ADCOBJ, handle);
		stobj = objtrack_handle_to_id(UPOBJ, handle);
		config_create_wire(c, b, adobj , stobj+0); 
		config_create_wire(c, b, stobj+0, stobj+1);
		config_create_wire(c, b, stobj+0, stobj+2);
		config_create_wire(c, b, stobj+1, stobj+4);	// AGC->ADPCM
		config_create_wire(c, b, stobj+4, stobj+5);
		config_create_wire(c, b, stobj+5, stobj+6);
		config_create_wire(c, b, stobj+1, stobj+7);	// AGC->MULAW
		config_create_wire(c, b, stobj+1, stobj+8);	// AGC->ALAW
		config_create_wire(c, b, adobj  , stobj+9);	// ADC -> DEC
		config_create_wire(c, b, stobj+9, stobj+3);	// DEC -> DTMF

		// Disable objects until used

		stobj = objtrack_handle_to_id(UPOBJ, handle);
		config_disable_object(vpb_c, b, stobj+6);	// PACKED_FIFO_UP
		config_disable_object(vpb_c, b, stobj+5);	// PACK
		config_disable_object(vpb_c, b, stobj+4);	// LIN2ADPCM
		config_disable_object(vpb_c, b, stobj+7);	// LIN2MULAW
		config_disable_object(vpb_c, b, stobj+8);	// LIN2ALAW

	}
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	CloseVPB
	AUTHOR..:	David Rowe
	DATE....:	10/2/98
	
	This function is called during system shutdown to free all of the
	memory allocated for DSP FIFOs during ConfigureVPB().

\*-------------------------------------------------------------------------*/

void CloseVPB(Comm *c, USHORT numboards)
//  Comm    *c;			ptr to comm object for VPB(s)
//	USHORT  numboards;	number of boards in system
{
	VPBREG	*v;			// ptr to reg info for current VPB
	int		b,ch;		// board and channel on board
	int		handle;		// handle of current  combination

	
	if (vpb_c->vpbreg(0)->model != VPB_V12PCI) {

		mprintf("Free-ing memory for DSP FIFOs..\n");

		for(b=0; b<numboards; b++) {
			v = c->vpbreg(b);

		        #ifdef LINUX
			// shut down host ec threads (if enabled)
			if (v->echo_mode == DSPFIFO_ECHO) {
				for(ch=0; ch<v->numch; ch++) {
					dspfifo_echo_stop_thread(v->ec[ch]);
				}
			}
			#endif

			config_stop(vpb_c, b);
			for(ch=0; ch<v->numch; ch++) {
				handle = mapdevtohndle(b,ch);
				mprintf("Closing FIFO: %d  brd: "
					"%d  ch: %d\n",handle,b,ch);
				delete v->txdf[ch];
				delete v->rxdf[ch];
			}
		}

		mprintf("DSP FIFO memory free-ed OK!\n");
	}
}

/*-------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_async_bayonne 
	AUTHOR..: David Rowe
	DATE....: 24/8/00
	
	Note: This function is only used in "bayonne" mode.  Please do not
	call in "regular" mode - see also vpb_start()

	This function returns an event from the specified VPB card.  If
	an event is available it returns VPB_OK, otherwise it returns
	VPB_NO_EVENTS.
	
	When this function returns, it should be called again as soon as 
	possible in a tight polling loop, as it is responsible for sequencing
	several time-sensitive operations (e.g. playing DTMF tones).

	Avoid placing blocking functions in the polling loop that calls
	this function, to ensure it is called regularly.  Once every 20ms
	is recommended.

        IMPORTANT: This function must be called regularly, as the driver uses
        this function to service time critical tasks in the driver.  This
        function must be polled regularly, even if no events are required for 
        the program.  Without it, other functions, such as play/record, will
        not work.

\*-------------------------------------------------------------------------*/

int WINAPI vpb_get_event_async_bayonne(
 			int board,    // VPB board number: 1,2,3..
			VPB_EVENT *ev // event
)
{
	word      mess[COMM_MAX_MESS];     //  message from VPB
	long	  l;
	int	  h,i;
	VPB_EVENT e;
	int       mask;

	board -= 1;
	e.type = -1;

	// make sure DSP is still with us
        vpb_c->CheckForAssert(board);

	// OK, read DSP event Q
	if (vpb_c->GetMessageVPB(board,mess)==OK) {
		//mprintf("Copped an event mess[1] %d\n", mess[1]);

		switch (mess[1]) {
		case DSP_PONG:
			mprintf("PONG");
			break;
		case DSP_ERROR:
			assert(0);
			break;
		case DSP_TONED:
			assert(mess[0] == DSP_LTONED);
			e.type = VPB_TONEDETECT;
			e.handle = objtrack_id_to_handle(CPOBJ, mess[2], 
							 board);
			e.data = mess[3];
			mask = VPB_MTONEDETECT;
			break;
		case DSP_TONEDOK:
			mprintf("tone OK\n");
			break;
		case DSP_TONEG:
		case DSP_CODEC_BKEN:
			vpbdial_process_event(mess, board);
			break;
		case DSP_CONFIG_AF:
			break;
		case DSP_DTMF:
			assert(mess[0] == DSP_LDTMF);
			h = objtrack_id_to_handle(DTMFOBJ, mess[2], board);
			e.type = VPB_DTMF;
			e.handle = h;
			e.data = mess[3];
			mask = VPB_MDTMF;
			break;
		case DSP_CONFIG_FUO:
			assert(mess[0] == DSP_LCONFIG_FDU);
			e.type = VPB_RECORD_OVERFLOW;
			e.handle = objtrack_id_to_handle(FIFOUPOBJ, mess[2], 
							 board);
			e.data = 0;
			mask = VPB_MRECORD_OVERFLOW;
			break;
		case DSP_CONFIG_FDU:
			assert(mess[0] == DSP_LCONFIG_FDU);
		       	e.handle = objtrack_id_to_handle(FIFODNOBJ, 
				       			 mess[2], 
				       			 board);
			if (playrec_underflow_valid(e.handle)) {
				e.type = VPB_PLAY_UNDERFLOW;
				e.data = 0;
				mask = VPB_MPLAY_UNDERFLOW;
			}
			break;
		case DSP_COMM_MOF:
			break;
		case DSP_CODEC_RING:
			assert(mess[0] == DSP_LCODEC_RING);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_RING;
			e.handle = h;
			e.data = 0;
			mask = VPB_MRING;
			break;
		case DSP_RING_OFF:
			assert(mess[0] == DSP_LRING_OFF);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_RING_OFF;
			e.handle = h;
			e.data = 0;
			mask = VPB_MRING_OFF;
			break;
		case DSP_DROP:
			assert(mess[0] == DSP_LDROP);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_DROP;
			e.handle = h;
			e.data = 0;
			mask = VPB_MDROP;
		break;
		case DSP_TONED_LOG:
			toned_debug_log(board, mess);
			break;
		case DSP_DEBUG_LONG:
			l = mess[3];
			l <<= 16;
			l+= mess[4];
			mprintf("debug long[%d] 0x%lx\n",mess[2],l);
			break;
		case DSP_DEBUG_ARRAY:
			mprintf("\ndebug array[%d]:",mess[2]);
			for(i=0; i<mess[3]; i++)
				mprintf("0x%04x ",mess[4+i]);
			break;
		case DSP_DEBUG_STACK:
			l = mess[3];
			mprintf("debug stack[%d] 0x%x\n",mess[2],(USHORT)l);
			break;
		case DSP_VOXON:
			assert(mess[0] == DSP_LVOXON);
			e.type = VPB_VOXON;
			e.handle = objtrack_id_to_handle(VOXOBJ, mess[2], 
							 board);
			e.data = GenerictimeGetTime();
			mask = VPB_MVOXON;
			break;
		case DSP_VOXOFF:
			assert(mess[0] == DSP_LVOXOFF);
			e.type = VPB_VOXOFF;
			e.handle = objtrack_id_to_handle(VOXOBJ, mess[2], 
							 board);
			e.data = GenerictimeGetTime();
			mask = VPB_MVOXOFF;
			break;
		case DSP_SPI_LOAD:
			long sum,num;
			float scale;
			scale = (float)40.0/mess[10];
			sum = mess[6];
			sum <<= 16;
			sum += mess[7];
			num = mess[8];
			num <<= 16;
			num += mess[9];
			mprintf("[%02d] sam: %3.2f MIPs max: %3.2f MIPs av:"
				"%3.2f MIPs\n", board,scale*mess[3],
				scale*mess[5],scale*(float)sum/num);
			break;
		case DSP_FIFODIS:
			assert(mess[0] == DSP_LFIFODIS);
			h = objtrack_id_to_handle(FIFOUPOBJ, mess[2], board);
			// DR 10/3/02 - not used anymore 
			// playrec_fifo_disabled(h);
			break;
		default:
			assert(0);		// shouldn't get here
		} // switch
	}

	// HOOKS INTO OTHER MODULES THAT NEED REGULAR CHECKS---------------

	// allow termination in dialling and tone generation
	vpbdial_check_arb();

	// OK, if event found, return event if unmasked

	if (e.type == -1)
		return VPB_NO_EVENTS;

        if (mask & vpb_dev[e.handle].Chnevtmsk) {
		memcpy(ev, &e, sizeof(VPB_EVENT));
		return VPB_OK;
	} 
	else
		return VPB_NO_EVENTS;
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	MonitorMessageQueue 
	AUTHOR..:	John Kostogiannis and David Rowe
	DATE....:	9/12/97
	
	This function monitors the message queues for each board in 
	the system.  This function is called periodically by the operating 
	system, it then polls each VPB for messages and processes the events.

	Note: This function is used in "regular" mode instead of
	vpb_get_event_async_bayonne(), which is used in "bayonne" mode
	for the same function.

\*-------------------------------------------------------------------------*/

void MonitorMessageQueue(void *data)
{
	USHORT i;
	int    mess_ind;
	word   message[COMM_MAX_MESS];		//  message from VPB

	try {
		
		while (!MMQ_flag) {
			for(i=0;i<numboards;i++)
			{	
				// get message from VPB
			
				vpb_c->CheckForAssert(i);
				while ((mess_ind = vpb_c->GetMessageVPB(i,message))==OK){
					ProcessVPBMessage(message, i);
				} 
			}	

			// HOOKS INTO OTHER MODULES THAT NEED REGULAR CHECKS------

			// check active timers

			vpbtimer_check_active_timers();

			// check async get digits for time outs

			digits_check_timers();

			// check call progress timers
			
			call_check_timer();

			// its pip to be square

			pip_pip(MESSEVENTDELAY);
			
			// allow termination in dialling and tone generation

			vpbdial_check_arb();

			GenericSleep(MESSEVENTDELAY);
		} // while(!MMQ_flag)
	}

	catch(Wobbly w){
		char s[MAX_STR];
		w.translate(s);
				
		if (w.file[0] != 0) 
			mprintf("exception caught: %s, %s, line = %d\n",s, w.file, w.line);
		else
			mprintf("exception caught: %s\n",s);

		//mprintf("Press any key to exit....\n");
		//assert(0);
	}

	// Signal MMQ kill function that MMQ is dead!
	MMQ_flag = 0;
	Generic_endthread();
}

/*-------------------------------------------------------------------------*\

	FUNCTION:	ProcessVPBMessage 
	AUTHOR..:	David Rowe
	DATE....:	5/4/98
	
	Given a message from a VPB, this function determines how to process it.
	
\*-------------------------------------------------------------------------*/

void ProcessVPBMessage(word mess[], USHORT board)
// word	   mess[]		message from VPB
// USHORT  board		number of board that generated message
{
	VPB_EVENT	e;
	long		l;
	int			h,i;
	
	//mprintf("Copped an event mess[1] %d\n", mess[1]);

	switch (mess[1]) {
		case DSP_PONG:
			mprintf("PONG");
		break;
		case DSP_ERROR:
			assert(0);
		break;
		case DSP_TONED:
			assert(mess[0] == DSP_LTONED);
			e.type = VPB_TONEDETECT;
			e.handle = objtrack_id_to_handle(CPOBJ, mess[2], board);
			e.data = mess[3];
			putevt(&e, VPB_MTONEDETECT);

			// send tone info to call progress detector

			call_new_tone(e.handle, e.data);
		break;
		case DSP_TONEDOK:
			mprintf("tone OK\n");
		break;
		case DSP_TONEG:
		case DSP_CODEC_BKEN:
			vpbdial_process_event(mess, board);
		break;
		break;
		case DSP_CONFIG_AF:
		break;
		case DSP_DTMF:
 			assert(mess[0] == DSP_LDTMF);
			if (vpb_c->vpbreg(0)->model == VPB_V12PCI)
				h = mapdevtohndle(board, mess[2]);
			else
				h = objtrack_id_to_handle(DTMFOBJ, mess[2], 
							  board);
			e.type = VPB_DTMF;
			e.handle = h;
			e.data = mess[3];
			putevt(&e, VPB_MDTMF);
			digits_new_digit(h, mess[3]);
			playrec_new_digit_record(h, mess[3]);
			playrec_new_digit_play(h, mess[3]);
		break;
		case DSP_CONFIG_FUO:
			assert(mess[0] == DSP_LCONFIG_FDU);
			e.type = VPB_RECORD_OVERFLOW;
			e.handle = objtrack_id_to_handle(FIFOUPOBJ, mess[2], board);
			e.data = 0;
			putevt(&e, VPB_MRECORD_OVERFLOW);
		break;
		case DSP_CONFIG_FDU:
			assert(mess[0] == DSP_LCONFIG_FDU);
			e.type = VPB_PLAY_UNDERFLOW;
			e.handle = objtrack_id_to_handle(FIFODNOBJ, mess[2], board);
			e.data = 0;
			if (playrec_underflow_valid(e.handle))
				putevt(&e, VPB_MPLAY_UNDERFLOW);
		break;
		case DSP_COMM_MOF:
		break;
		case DSP_CODEC_RING:
			assert(mess[0] == DSP_LCODEC_RING);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_RING;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MRING);
		break;
		case DSP_RING_OFF:
			assert(mess[0] == DSP_LRING_OFF);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_RING_OFF;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MRING_OFF);
		break;
		case DSP_DROP:
			assert(mess[0] == DSP_LDROP);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_DROP;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MDROP);
		break;
		case DSP_TONED_LOG:
			toned_debug_log(board, mess);
		break;
		case DSP_DEBUG_LONG:
			l = mess[3];
			l <<= 16;
			l+= mess[4];
			mprintf("debug long[%d] %ld\n",mess[2],l);
		break;
		case DSP_DEBUG_ARRAY:
			mprintf("\ndebug array[%d]:",mess[2]);
			for(i=0; i<mess[3]; i++)
				mprintf("0x%04x ",mess[4+i]);
		break;
		case DSP_DEBUG_STACK:
			l = mess[3];
			mprintf("debug stack[%d] 0x%x\n",mess[2],(USHORT)l);
		break;
		case DSP_VOXON:
			assert(mess[0] == DSP_LVOXON);
			e.type = VPB_VOXON;
			e.handle = objtrack_id_to_handle(VOXOBJ, mess[2], board);
			e.data = GenerictimeGetTime();
			putevt(&e, VPB_MVOXON);
		break;
		case DSP_VOXOFF:
			assert(mess[0] == DSP_LVOXOFF);
			e.type = VPB_VOXOFF;
			e.handle = objtrack_id_to_handle(VOXOBJ, mess[2], board);
			e.data = GenerictimeGetTime();
			putevt(&e, VPB_MVOXOFF);
		break;
		case DSP_SPI_LOAD:
			long sum,num;
			float scale;
			scale = (float)40.0/mess[10];
			sum = mess[6];
			sum <<= 16;
			sum += mess[7];
			num = mess[8];
			num <<= 16;
			num += mess[9];
			mprintf("[%02d] sam: %3.2f MIPs max: %3.2f MIPs av: %3.2f MIPs\n",
					board,scale*mess[3],scale*mess[5],scale*(float)sum/num);
		break;
		case DSP_FIFODIS:
			//mprintf("DSP_FIFODIS");
			assert(mess[0] == DSP_LFIFODIS);
			h = objtrack_id_to_handle(FIFOUPOBJ, mess[2], board);
			// DR 10/3/02 - not used naymore
			// playrec_fifo_disabled(h);
		break;
		case DSP_CODEC_HKOFF:
			assert(mess[0] == DSP_LCODEC_HKOFF);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_STATION_OFFHOOK;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MSTATION_OFFHOOK);
			break;
		case DSP_CODEC_HKON:
			assert(mess[0] == DSP_LCODEC_HKON);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_STATION_ONHOOK;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MSTATION_ONHOOK);
			break;
		case DSP_CODEC_FLASH:
			assert(mess[0] == DSP_LCODEC_FLASH);
			h = mapdevtohndle(board, mess[2]);
			e.type = VPB_STATION_FLASH;
			e.handle = h;
			e.data = 0;
			putevt(&e, VPB_MSTATION_FLASH);
			break;
		default:
			assert(0);		// shouldn't get here
	} // switch
}

/*-------------------------------------------------------------------------*\
	
	FUNCTION:	StartMonitorMessageQueue 
	AUTHOR..:	John Kostogiannis
	DATE....:	9/12/97
	
	This function configures and intialises the monitoring of the message 
	queues for each board in the system. 
	
\*-------------------------------------------------------------------------*/

void StartMonitorMessageQueue()
{	
	MMQ_flag = 0;
	Generic_beginthread(MonitorMessageQueue, 0, NULL);
	mprintf("timer callback started OK!\n");
}

/*-------------------------------------------------------------------------*\
	
	FUNCTION:	KillMonitorMessageQueue 
	AUTHOR..:	John Kostogiannis
	DATE....:	9/12/97
	
	This function kills the monitoring of the message 
	queue for the system. 
	
\*-------------------------------------------------------------------------*/

void KillMonitorMessageQueue()
{
	// signal MMQ to exit
	MMQ_flag = 1;
	
	// wait for MMQ to exit
	while(MMQ_flag)
		GenericSleep(MESSEVENTDELAY);
}	
	
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_event_callback
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Sets the event callback function for each channel.  By default no
	event callback functions are set, in this case events are posted
	to the API event Q.  If this function has been called with a
	non-null function pointer, the event callback funcion is called with
	the event as an argument rather than posting the event to the API
	Q.
	
	The user defined event callback function should be of the form:

	void event_callback(VPB_EVENT *e);

	To disable the event callback, call this function with a NULL 
	pointer for the function argument.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_event_callback(int handle, void (WINAPI *event_callback)(VPB_EVENT *e, void *context), void *context)
//	int		handle										device handle
//	void (*event_callback)(VPB_EVENT *e, void *context)	user defined event callback function
//  void    *context									user defined context info
{
	try {
		ValidHandleCheck(handle);
		vpb_dev[handle].event_callback = event_callback;
		vpb_dev[handle].context = context;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_event_callback"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_event_mask
	AUTHOR..: John Kostogiannis
	DATE....: 4/12/97

	Function to set the event mask for each device.  Each call to this
	function overwrites the previous event mask.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_event_mask(int handle, unsigned short mask)
//	int				handle	device handle
//	unsigned short	mask	new mask to assign to this device		
{
	try {
		ValidHandleCheck(handle);
		vpb_dev[handle].Chnevtmsk = mask;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_event_mask"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_mask
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to return the current event mask for a given device.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_event_mask(int handle)
//	int				handle	device handle
{
	
	try {
		ValidHandleCheck(handle);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_event_mask"));
	}
	
	return(vpb_dev[handle].Chnevtmsk);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_enable_event
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to unmask (enable) only the specified events while leaving the
	rest of the event masks unchanged.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_enable_event(int handle, unsigned short mask)
//	int				handle	device handle
//	unsigned short	mask	event(s) to unmask		
{
	try {
		ValidHandleCheck(handle);
		vpb_dev[handle].Chnevtmsk |= mask;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_enable_event"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_disable_event
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to mask out (disable) specified events only, while leaving the
	rest of the event masks unchanged.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_disable_event(int handle, unsigned short mask)
//	int				handle	device handle
//	unsigned short	mask	evt(s) to mask
{
	try {
		ValidHandleCheck(handle);
		vpb_dev[handle].Chnevtmsk &= ~mask;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_disable_event"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_put_event
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to place an event on the API event queue exported API version.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_put_event(VPB_EVENT *e)
//	VPB_EVENT	*e		// event to place on API Q
{

	try {
		// prevent App and poll timer calling at the same time

		GenericEnterCriticalSection(&PutEvtSect);
	
		// otherwise place event on Q
	
		APIQ->Write((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
		vpb_dev[e->handle].APIQ->Write((USHORT*)e, 
					       SIZE_OF_VPB_EVENT_WORDS);

		GenericLeaveCriticalSection(&PutEvtSect);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_put_event"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: putevt
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to place an event on the API event queue, local version.

\*--------------------------------------------------------------------------*/

int putevt(VPB_EVENT *e, USHORT mask)
//	VPB_EVENT	*e		// event to place on API Q
//	USHORT		mask	// mask (set to 0 if not required)
{
	USHORT		words;
	
	if ((mask==0) || (mask & vpb_dev[e->handle].Chnevtmsk))
	
		if (vpb_dev[e->handle].DevHndles == VPB_ON) {

			// prevent App and poll timer calling at the same time

			GenericEnterCriticalSection(&PutEvtSect);

			// determine if callback function to be used

			if (vpb_dev[e->handle].event_callback == NULL) {
				
				// ring a few alarm bells if API queue nearly 
			        // full

				APIQ->HowFull(&words);

				// if event queue fills it will just have no
				// more events added

				if (words > (APIQUEUESIZE-2)*SIZE_OF_VPB_EVENT_WORDS) {
					//mprintf("Oops - Event Q full!\n");
				}
	
				// otherwise place event on Q
		
				int ret;
				ret = APIQ->Write((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
				if (ret != OK) {
					//mprintf("Oops - problem writing to global Event Q!\n");
				}
				ret = vpb_dev[e->handle].APIQ->Write((USHORT*)e,
								     SIZE_OF_VPB_EVENT_WORDS);
				if (ret != OK) {
					mprintf("Oops - problem writing to ch Event Q!\n");
					mprintf("h = %d type = %d ret = %d \n", e->handle, e->type, ret);
				}
			}
			else {
				start_event_callback_thread(e);
			}
			GenericLeaveCriticalSection(&PutEvtSect);
		}

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: start_event_callback_thread
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	This function starts a new thread to call the event callback
	function.  A new thread is required as we dont want the MMQ
	thread to be stuck whle the callback function finishes 
	processing.

	The callback function is responsble for deleting the context
	data.

\*--------------------------------------------------------------------------*/

void start_event_callback_thread(VPB_EVENT *e)
//	VPB_EVENT	*e;		event to pass to callback
{
	VPB_EVENT *ev = new VPB_EVENT;
	ev->handle = e->handle;
	ev->type = e->type;
	ev->data = e->data;
	ev->data1 = e->data1;

	Generic_beginthread(event_callback_thread, 0, (void*)ev);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: event_callback_thread
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Helper function called when event callback thread starts, calls
	the user defined event callback function and deletes the context
	info when finished so the user doesnt have to worry.

\*--------------------------------------------------------------------------*/

void event_callback_thread(void *data)
{
	VPB_EVENT	*e = (VPB_EVENT*)data;

	vpb_dev[e->handle].event_callback(e, vpb_dev[e->handle].context);
	delete e;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_async
	AUTHOR..: David Rowe
	DATE....: 7/2/98

	Function to get an event from the API event queue.  Returns VPB_OK
	if an event available, other wise VPB_NO_EVENTS if Q empty.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_event_async(VPB_EVENT *e)
//	VPB_EVENT	*e		// event to place on API Q
{
	int			ret;

	// try to read an event

	ret = APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
	if (ret == OK)
		return(VPB_OK);
	else {
		return(VPB_NO_EVENTS);
		e->type = -1;
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_sync
	AUTHOR..: David Rowe
	DATE....: 25/3/98

	This function waits until an event is available on the API Q, then 
	returns.  The thread is put to sleep until the event becomes
	available.

	Function returns VPB_OK if event returned, else VPB_TIME_OUT if
	sync function times out.  Use tme_out = 0 for no timeout.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_event_sync(VPB_EVENT *e, unsigned int time_out)
//	VPB_EVENT	*e		// event to place on API Q
{
	int				ret;
	unsigned int	time=0;

	ret = APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
	while(ret != OK) {
		GenericSleep(sleepms);

		// check for time out

		time += sleepms;
		if (time_out)
			if (time >= time_out) {
				e->type = -1;
				return(VPB_TIME_OUT);
			}

		ret = APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
	}

	// event found

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_ch_sync
	AUTHOR..: David Rowe
	DATE....: 7/8/98

	This function waits until an event is available on the API Q, then 
	returns.  The thread is put to sleep until the event becomes
	available.  This version only returns events from a specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_event_ch_sync(int h, VPB_EVENT *e, unsigned int time_out)
{
	int				ret;
	unsigned int	time=0;

	try {
		ValidHandleCheck(h);

		ret = vpb_dev[h].APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
		while(ret != OK) {
			GenericSleep(sleepms);

			// check for time out

			time += sleepms;
			if (time_out)
				if (time >= time_out) {
					e->type = -1;
					return(VPB_TIME_OUT);
				}

			ret = vpb_dev[h].APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
		}

	}
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_event_ch_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_event_ch_async
	AUTHOR..: David Rowe
	DATE....: 7/8/98

	This reads an event on a channels event queue if available.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_event_ch_async(int h, VPB_EVENT *e)
{
	int			ret;

	try {
		ValidHandleCheck(h);
		ret = vpb_dev[h].APIQ->Read((USHORT*)e, SIZE_OF_VPB_EVENT_WORDS);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_event_ch_aync"));
	}
	
	if (ret == OK)
		return(VPB_OK);
	else {
		return(VPB_NO_EVENTS);
		e->type = -1;
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_sethook_async
	AUTHOR..: David Rowe
	DATE....: 10/2/98

	Function to set the hook status of a particular chnnel device.  This
	version returns immediately.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_sethook_async(int chdev, int hookstate)
{
	USHORT	b,ch;		// board and channel for this handle

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (hookstate & VPB_OFFHOOK) {
			offhook[2] = ch;
			if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
			    vpb_set_codec_reg(chdev, 0x12, 0x7c);
			// Enable off hook through codec dir reg.
			vpb_c->PutMessageVPB(b,offhook);
		}
		else {
			onhook[2] = ch;
			vpb_c->PutMessageVPB(b,onhook);
			if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
			    vpb_set_codec_reg(chdev, 0x12, 0x1c);
			// Disable off hook through codec dir reg.
		}
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_sethook_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_sethook_sync
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Function to set the hook status of a particular chnnel device.  This
	version returns after the hook status has been changed.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_sethook_sync(int chdev, int hookstate)
{
	USHORT	b,ch;		// board and channel for this handle

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (hookstate & VPB_OFFHOOK) {
			offhook[2] = ch;
			if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
			    vpb_set_codec_reg(chdev, 0x12, 0x7c);
			    // Enable off hook through codec dir reg.
			vpb_c->PutMessageVPB(b,offhook);
		}
		else {
			onhook[2] = ch;
			vpb_c->PutMessageVPB(b,onhook);
			if ((vpbreg_get_model() == VPB_V4PCI)||(vpbreg_get_model() == VPB_V4LOG))
			    vpb_set_codec_reg(chdev, 0x12, 0x1c);
			    // Disable off hook through codec dir reg.
		}

		// delay determined by experiment to give hardware enough time
		// to take line on/off hook.  This is a bit slow due to speed
		// of link through codecs.

		GenericSleep(100);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_sethook_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_station_sync
	AUTHOR..: David Rowe
	DATE....: 17/5/02

	Function to set the status (on or off) of a station port channel 
	device.  This version returns after the status has been changed.

	NOTE: DR 22/8/02 - this function is not normally required, as hook
	detector logic has been modified to automatically turn off/on
	station audio with hook switch.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_station_sync(int chdev, int state)
{
	USHORT	b,ch;	       
	word    mess[PC_LSTATION_ON];

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (state == VPB_STATION_ON) {
			mess[0] = PC_LSTATION_ON;
			mess[1] = PC_STATION_ON;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}
		else {
			mess[0] = PC_LSTATION_OFF;
			mess[1] = PC_STATION_OFF;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,onhook);
		}

		GenericSleep(100);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_station_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_ring_station_async
	AUTHOR..: David Rowe
	DATE....: 17/5/02

	Function to start/stop a station port ringing.  This version rings
	stations with a pre-defined cadence.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_ring_station_async(int chdev, int state)
{
	USHORT	b,ch;	       
	word    mess[PC_LRING_ON];

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (state == VPB_RING_STATION_ON) {
			mess[0] = PC_LRING_ON;
			mess[1] = PC_RING_ON;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}
		else {
			mess[0] = PC_LRING_OFF;
			mess[1] = PC_RING_OFF;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_ring_station_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_ring_station_sync
	AUTHOR..: David Rowe
	DATE....: 17/5/02

	Function to start/stop a station port ringing.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_ring_station_sync(int chdev, int state)
{
	USHORT	b,ch;	       
	word    mess[PC_LRING_ON];

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (state == VPB_RING_STATION_ON) {
			mess[0] = PC_LRING_ON;
			mess[1] = PC_RING_ON;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}
		else {
			mess[0] = PC_LRING_OFF;
			mess[1] = PC_RING_OFF;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}

		//GenericSleep(100);	// not required
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_ring_station_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_user_ring_station_sync
	AUTHOR..: David Rowe
	DATE....: 17/5/02

	Function to start/stop a station port ringing.  This version allows
	the caller (user) to define the cadence.  Station will ring
	continuously when called with state==1, until called with state==0.

	Care should be taken to ring no more than that 4 
	stations at any one time, to prevent overloading the OpenSwitch12
	ring generator.  Excess load of the ring generator will not result
	in permanent damage, however the station power supply to shut down,
	requiring a power down of the OpenSwitch12 (ie switching the PC off)
	to reset.

	This function should not be used in conjunction with
	vpb_ring_station_sync, the cadenced ring generator function, as the
	driver will get confused!

\*--------------------------------------------------------------------------*/

int WINAPI vpb_user_ring_station_sync(int chdev, int state)
{
	USHORT	b,ch;	       
	word    mess[PC_LURING_ON];

	try {
		ValidHandleCheck(chdev);
		maphndletodev(chdev, &b, &ch);
		if (state == VPB_RING_STATION_ON) {
			mess[0] = PC_LURING_ON;
			mess[1] = PC_URING_ON;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}
		else {
			mess[0] = PC_LURING_OFF;
			mess[1] = PC_URING_OFF;
			mess[2] = ch;
			vpb_c->PutMessageVPB(b,mess);
		}

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_user_ring_station_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_sleep
	AUTHOR..: David Rowe
	DATE....: 15/2/98

	Function to put the current thread to sleep for a certain time in ms.
        Used during testing to reduce the amount of time program spends in main
	polling loop so that the computational load of the timer callback
	can be measured.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_sleep(long time)
//	long	timeout		amount of time to sleep in ms
{
	GenericSleep(time);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_adpcm_open
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Sets up the state variables for ADPCM opertaions.  Must be called before 
	any other ADPCM functions.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_adpcm_open(void **pv)
{
	try {
		adpcm_open(pv);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_adpcm_open"));
	}
	
	return(VPB_OK);

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_adpcm_close
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Closes down the ADPCM state variables, freeing memory.  Call after all
	ADPCM operations have finished.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_adpcm_close(void *pv)
{
	adpcm_close(pv);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_adpcm_reset
	AUTHOR..: David Rowe
	DATE....: 7/8/01

	Resets the ADPCM state variables.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_adpcm_reset(void *pv)
{
	adpcm_reset(pv);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_adpcm_decode
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Converts a buffer of compressed OKI ADPCM samples to a buffer of 16 bit
	linear samples.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_adpcm_decode(
	void			*pv,	   // ADPCM state variables
	short			linear[],  // linear output samples
	unsigned short  *size_lin,	   // number of 16 bit samples out
	char			adpcm[],   // packed ADPCM samples
	unsigned short	size_adpcm	   // number of bytes in (must be even)
)
{
	USHORT	*codes;
	
	try {
		if (size_adpcm%2)
			throw Wobbly(VPBAPI_ADPCM_SIZE_ADPCM_MUST_BE_EVEN);
		CheckNew(codes = new word[size_adpcm*2]);
	
		adpcm_unpack(codes, (USHORT*)adpcm, size_adpcm/2);
		adpcm_decode(pv, linear, codes, size_adpcm*2);
		*size_lin = size_adpcm*2;

		delete codes;
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_adpcm_decode"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_adpcm_encode
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Converts a buffer of 16 bit linear samples to a buffer of compressed 
	OKI ADPCM samples.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_adpcm_encode(
	void		*pv,	     // ADPCM state variables
	char		adpcm[],     // output packed ADPCM samples
	unsigned short	*size_adpcm, // number of bytes out
	short		linear[],    // linear input samples
	unsigned short  size_linear  // number of 16 bit samples in
)
{
	USHORT	*codes;
	
	try {
		CheckNew(codes = new word[size_linear]);
	
		adpcm_encode(pv, codes, linear, size_linear);
		adpcm_pack(codes, (USHORT*)adpcm, size_linear/4);
		*size_adpcm = size_linear/2;

		delete codes;
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_adpcm_encode"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_comp_load
	AUTHOR..: David Rowe
	DATE....: 25/6/98

	Interrogates the DSP to determine the computational loading.
	Causes a comp load statement to be printed if mprintf's are
	enabled.

	Dont expose the function to the punters, keep it as undocumented
	for our use only.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_comp_load(unsigned short board)
{
	word	m[PC_LSPI_LOAD];
	
	try {
		m[0] = PC_LSPI_LOAD;
		m[1] = PC_SPI_LOAD;
		vpb_c->PutMessageVPB(board, m);
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_comp_load"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_force_adapt_on
	AUTHOR..: David Rowe
	DATE....: 25/6/98

	Forces the echo cancellers to adapt all of the time, ie switches
	off near end speech detector.  Used to determine maximum 
	computational load.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_force_adapt_on()
{
	word	m[PC_LECHO_FORCEON];
	int		i,id;
	USHORT	b,ch;		// board and channel for this handle
	
	try {
		m[0] = PC_LECHO_FORCEON;
		m[1] = PC_ECHO_FORCEON;

		for(i=0; i<Totalchns; i++) {
			maphndletodev(i, &b, &ch);
			id = objtrack_handle_to_id(ECHOOBJ, i);
			m[2] = id;
			vpb_c->PutMessageVPB(b, m);
		}
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_force_adapt_on"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_force_adapt_off
	AUTHOR..: David Rowe
	DATE....: 25/6/98

	Stops forced adaptation.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_force_adapt_off()
{
	word	m[PC_LECHO_FORCEOFF];
	int		i,id;
	USHORT	b,ch;		// board and channel for this handle

	try {
		m[0] = PC_LECHO_FORCEOFF;
		m[1] = PC_ECHO_FORCEOFF;

		for(i=0; i<Totalchns; i++) {
			maphndletodev(i, &b, &ch);
			id = objtrack_handle_to_id(ECHOOBJ, i);
			m[2] = id;
			vpb_c->PutMessageVPB(b, m);
		}
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_force_adapt_off"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_enable
	AUTHOR..: David Rowe
	DATE....: 9/9/98

	Enables the echo cancellers.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_enable()
{
	word	m[PC_LECHO_ENABLE];
	int		i,id;
	USHORT	b,ch;		// board and channel for this handle
	
	try {
		m[0] = PC_LECHO_ENABLE;
		m[1] = PC_ECHO_ENABLE;

		for(i=0; i<Totalchns; i++) {
			maphndletodev(i, &b, &ch);
			id = objtrack_handle_to_id(ECHOOBJ, i);
			m[2] = id;
			vpb_c->PutMessageVPB(b, m);
		}
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_enable"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_disable
	AUTHOR..: David Rowe
	DATE....: 9/9/98

	Disables the echo cancellers, but keeps adaptation on.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_disable()
{
	word	m[PC_LECHO_DISABLE];
	int		i,id;
	USHORT	b,ch;		// board and channel for this handle
	
	try {
		m[0] = PC_LECHO_DISABLE;
		m[1] = PC_ECHO_DISABLE;

		for(i=0; i<Totalchns; i++) {
			maphndletodev(i, &b, &ch);
			id = objtrack_handle_to_id(ECHOOBJ, i);
			m[2] = id;
			vpb_c->PutMessageVPB(b, m);
		}
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_disable"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_get_sup_thresh
	AUTHOR..: David Rowe
	DATE....: 24/9/01

	Gets the current echo suppressor threshold.  This threshold is
	the same for all channels on all cards.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_get_sup_thresh(short *thresh)
{
	
	try {
		VPBREG *v=vpb_c->vpbreg(0);		
		long   unsigned agthres;

		// just read value from first card as all cards have same value
		coff_get_address(v->firm, "_gthres", &agthres);
		vpb_c->hip(0)->ReadDspSram(0, (USHORT)agthres, 1, 
					   (word*)thresh);
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_get_sup_thresh"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_echo_canc_set_sup_thresh
	AUTHOR..: David Rowe
	DATE....: 24/9/01

	Sets the current echo suppressor threshold.  This threshold is
	the same for all channels on all cards.

	0x1000 (4096) -18dB
	0x800  (2048) -24dB
	0      0      no supressor

\*--------------------------------------------------------------------------*/

int WINAPI vpb_echo_canc_set_sup_thresh(short *thresh)
{
	
	try {
		int    b;
		VPBREG *v=vpb_c->vpbreg(0);		
		long   unsigned agthres;

		coff_get_address(v->firm, "_gthres", &agthres);
		for(b=0; b<numboards; b++) {
			vpb_c->hip(b)->WriteDspSram(b, (USHORT)agthres, 1, 
						    (word*)thresh);
		}
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_echo_canc_set_sup_thresh"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_model
	AUTHOR..: David Rowe
	DATE....: 11/9/98

	Returns a string containing the VPB model, driver should be started
	first by at least one call to vpb_open().

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_model(char *s) {

	switch(vpb_c->vpbreg(0)->model)
       	{
		case VPB_VPB4:
			strcpy(s, "VPB4");
		        break;
			
		case VPB_VPB8L:
			strcpy(s, "VPB8L");
		        break;
			
		case VPB_V4LOG:
		        strcpy(s, "V4LOG");
			break;

		case VPB_V4PCI:
		        strcpy(s, "V4PCI");
			break;

		case VPB_V6PCI:
		        strcpy(s, "V6PCI");
		        break;
										
		case VPB_V12PCI:
			strcpy(s, "V12PCI");
			break;
		default:
			assert(0);
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_ports_per_card
	AUTHOR..: David Rowe
	DATE....: 3/10/02

	Returns the number of ports on each card.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_ports_per_card() {

	switch(vpb_c->vpbreg(0)->model) {
		case VPB_VPB4:
		  return 4;
		  break;
		case VPB_VPB8L:
		  return 8;
		  break;
		case VPB_V6PCI:
		  return 12;
		  break;
		case VPB_V12PCI:
		  return 12;
		  break;
		default:
		  assert(0);
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_codec_reg
	AUTHOR..: David Rowe
	DATE....: 9/2/01

	Used to set a register of the TS5070 codecs, allows driver-level config
	of codec, rather than hard-coding in DSP firmware.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_set_codec_reg(int chdev,    // channel handle
		              USHORT addr,  // 8-bit address of codec register
		              USHORT data   // 8-bit data to write to register
)
{
	set_codec_reg(chdev, addr, data, vpb_c);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: set_codec_reg
	AUTHOR..: David Rowe
	DATE....: 9/2/01

	Internal version of set_codec_reg().

\*--------------------------------------------------------------------------*/

void set_codec_reg(int    chdev, // channel handle
		   USHORT addr,  // 8-bit address of codec register
		   USHORT data,  // 8-bit data to write to register
		   Comm   *c
)
{
	USHORT	b,ch;		// board and channel for this handle
	word	m[PC_LECHO_DISABLE];
        
	maphndletodev(chdev, &b, &ch);
	m[0] = PC_LCODEC_GEN;
	m[1] = PC_CODEC_GEN;
	m[2] = ch;
	m[3] = addr;
	m[4] = data;
	c->PutMessageVPB(b,m);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_num_cards
	AUTHOR..: David Rowe
	DATE....: 24/7/01

	Returns the number of cards in the system.  In current implemention
	need to call vpb_open() or vpb_start() before calling this function.
	This means e.g. one channel should be opened before determining how 
	many cards there are! This is very silly and should be fixed some 
	time!

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_num_cards()
{
    assert(Init);
    return numboards;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_bridge
	AUTHOR..: David Rowe
	DATE....: 27/2/02

	Bridges two ports using the driver, to allow 2-way conferencing. The
	bridging is implemented at a low level (firmware and/or hardware) by 
	the driver to ensure a low delay connection between the ports.  Note
	that only ports on the same physical card can be bridged using this
	method.

	Two bridging resources are available on the V4PCI (1 & 2).

	For 3 or more way conferencing (or bridging between ports on different
	cards) use "software bridging" using the play and record buf API 
	functions.  This results in a higher delay (20ms or so), however this 
	does not cause a problem due to the echo canceller on each VPB port.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_bridge(
		       int h1,      // handle of first channel
		       int h2,      // handle of second channel
		       int mode,    // VPB_BRIDGE_ON | VPB_BRIDGE_OFF
		       int resource // bridge resource 1 or 2 (V4PCI only)
)
{
	USHORT	b1,ch1,b2,ch2;		
	word	m[PC_LSPI_BRIDGE];
        
	try {
	  // can't bridge a port to itself
	  if (h1 == h2)
	    throw Wobbly(VPBAPI_BRIDGE_HANDLES_IDENTICAL);

	  maphndletodev(h1, &b1, &ch1);
	  maphndletodev(h2, &b2, &ch2);
	
	  // can't bridge across boards (yet!)
	  if (b1 != b2)
	    throw Wobbly(VPBAPI_BRIDGE_DIFFERENT_BOARDS);
	
//	  printf("model = %d\n",model);
	  switch(model) {
	  case VPB_VPB4:
		  // only two bridge resources available on V4PCI
		  assert((resource == 1) || (resource == 2));

 	  case VPB_V12PCI:

		  if (mode == VPB_BRIDGE_ON) { 
			  m[0] = PC_LSPI_BRIDGE;
			  m[1] = PC_SPI_BRIDGE;
		  }
		  else {
			  m[0] = PC_LSPI_BRIDGE_OFF;
			  m[1] = PC_SPI_BRIDGE_OFF;
		  }
		  m[2] = ch1;
		  m[3] = ch2;
		  m[4] = resource;
		  vpb_c->PutMessageVPB(b1,m);
		  break;
	  default:
		  assert(0);
		  break;
	  }

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_bridge"));
	}
	
	return(VPB_OK);
	
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_sbridge
	AUTHOR..: Ben Kramer
	DATE....: 27/02/03

	Software bridge for OpenSwitch boards. Can bridge between boards.

\*--------------------------------------------------------------------------*/
int WINAPI vpb_sbridge(
		       int h1,      // handle of first channel
		       int h2,      // handle of second channel
		       int mode    // VPB_BRIDGE_ON | VPB_BRIDGE_OFF
)
{
	USHORT	b1,ch1,b2,ch2;		
	word	m[PC_LSPI_SBRIDGE];
        
	try {
	  // can't bridge a port to itself
	  if (h1 == h2)
	    throw Wobbly(VPBAPI_BRIDGE_HANDLES_IDENTICAL);

	  maphndletodev(h1, &b1, &ch1);
	  maphndletodev(h2, &b2, &ch2);
	
	
//	  printf("model = %d\n",model);
	  switch(model) {
	  case VPB_VPB4:
		throw Wobbly(VPBAPI_CONF_BOARD_NOT_SUPPORTED);

 	  case VPB_V12PCI:

		  if (mode == VPB_BRIDGE_ON) { 
			  m[0] = PC_LSPI_SBRIDGE;
			  m[1] = PC_SPI_SBRIDGE;
		  }
		  else {
			  m[0] = PC_LSPI_SBRIDGE_OFF;
			  m[1] = PC_SPI_SBRIDGE_OFF;
		  }
		  m[2] = ch1;
		  m[3] = b2;
		  m[4] = ch2;
		  vpb_c->PutMessageVPB(b1,m);
		  break;
	  default:
		  assert(0);
		  break;
	  }

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_bridge"));
	}
	
	return(VPB_OK);
	
}


/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_conf
	AUTHOR..: Ben Kramer
	DATE....: 26/02/03

	Multi party conferencing. Can also be used to bridge boards.
	There is a maximum of 10 conferences with a max of 10 people
	each.

\*--------------------------------------------------------------------------*/
int WINAPI vpb_conf(
		int h,		// Handle of channel/port
		int resource,	// Conference/party resource to use (0-9)
	       	int mode	// VPB_CONF_JOIN,VPB_VONF_LEAVE
)
{
	USHORT	b,ch;		
	word	m[PC_LCONF_JOIN];
        
	try {

	  maphndletodev(h, &b, &ch);
	
	
//	  printf("model = %d\n",model);
	  switch(model) {

 	  case VPB_V12PCI:

		if (mode && VPB_CONF_JOIN) { 
//	  printf("Joining conf %d to %d\n",ch,resource);
			m[0] = PC_LCONF_JOIN;
			m[1] = PC_CONF_JOIN;
			m[2] = ch;
			m[3] = resource;
		}
		else {
//	  printf("Leaving conf %d \n",ch);
			m[0] = PC_LCONF_LEAVE;
			m[1] = PC_CONF_LEAVE;
			m[2] = ch;
			m[3] = 0;
		}
		vpb_c->PutMessageVPB(b,m);
		break;

	  default:
		throw Wobbly(VPBAPI_CONF_BOARD_NOT_SUPPORTED);
		break;
	  }

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_conf"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_watchdog_enable
	AUTHOR..: David Rowe
	DATE....: 27/8/02

	Enables (enable == 1), or disables (enable == 0) V12PCI watchdog
	timer.  There is one watchdog timer per board, so only one call
	per board is required, any handle h from that board can be used.

	Once enabled, it must be reset at
	least once every 8 seconds by vpb_watchdog_reset.  If it is not
	reset the watchdog timer will fire.  Once fired, the hardware will
        assume the PC has crashed, and the first 4 station ports will be 
	connected to the first 4 loop ports to preserve basic phone
	functionality.

	Use of the watchdog timer assumes a mixed station/loop 
	configuration.  Do not use the watchdog timer when the card is
	configured as all loop or all station cards.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_watchdog_enable(int h, int enable) {
	word	m[PC_LWD_ENABLE];
	USHORT	b,ch;		// board and channel for this handle
	
	try {
		m[0] = PC_LWD_ENABLE;
		m[1] = PC_WD_ENABLE;
		m[2] = enable;

		maphndletodev(h, &b, &ch);
		vpb_c->PutMessageVPB(b, m);
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_watchdog_enable"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_watchdog_reset
	AUTHOR..: David Rowe
	DATE....: 27/8/02

	Resets watchdog timer.  Call at least once every 8 seconds to prevent
	watchdog timer firing.  Does nothing when watchdog timer is disabled.

	There is one watchdog timer per board, so only one call
	per board is required, any handle h from that board can be used.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_watchdog_reset(int h) {
	word	m[PC_LWD_RESET];
	USHORT	b,ch;		// board and channel for this handle
	
	try {
		m[0] = PC_LWD_RESET;
		m[1] = PC_WD_RESET;

		maphndletodev(h, &b, &ch);
		vpb_c->PutMessageVPB(b, m);
	}
	
	catch(Wobbly w){
		return(RunTimeError(w,"vpb_watchdog_reset"));
	}
	
	return(VPB_OK);
}


/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_card_info()
	AUTHOR..: Peter Wintulich
	DATE....: 07/2/03

	Returns the card details in a structure VPB_CARD_INFO.
	Includes:	Model name,
			Serial number, 
			Revision number,
			Manufacture Date
			
	All strings are null terminated.

	Note: Only supported under Linux at this time!
	
\*---------------------------------------------------------------------------*/

int WINAPI vpb_get_card_info(
			int 		board,	// Board number
	       		VPB_CARD_INFO 	*detail // Card Info Structure to fill.
)
{
	#ifdef FREEBSD
	// DR 20/2/03 - trap use under OS other than Linux and Windows
	assert(0);
	#endif

    try {
    if(detail !=NULL)
    {
	    //vpb_get_model(char model);	// only dose first card!!!!
	switch(vpb_c->vpbreg(board)->model)
       	{
	    case VPB_VPB4:
	        strcpy(detail->model, "VPB4");
	        break;

	    case VPB_VPB8L:
	        strcpy(detail->model, "VPB8L");
	        break;

	    case VPB_V4LOG:
	        strcpy(detail->model, "V4LOG");
	        break;

	    case VPB_V4PCI:
	        strcpy(detail->model, "V4PCI");
	        break;

	    case VPB_V6PCI:
	        strcpy(detail->model, "V6PCI");
	        break;

	    case VPB_V12PCI:
		strcpy(detail->model, "V12PCI");
		break;
		
	    default:
	        strcpy(detail->model,"");
	}
	     
	strcpy(detail->date, vpb_c->vpbreg(board)->mdate); 
	strcpy(detail->rev, vpb_c->vpbreg(board)->revision);
	strcpy(detail->sn, vpb_c->vpbreg(board)->serial_n);
    }
  }
  
  catch(Wobbly w){
        return(RunTimeError(w,"vpb_get_card_info"));
  }
	
  
  return(VPB_OK);	
}
